//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.12

import wesual.Ui 1.0

/*!
\qmltype     UiCommandLink
\inqmlmodule we.captivo.Ui
\since       we.captivo.Ui 1.0
\inherits    QtQuick2::Item
\ingroup     captivo-ui-controls

\brief A clickable link.

The UiCommandLink component provides a text link similar to a hyperlink in
a web page. The \l {UiCommandLink::}{clicked} signal is emitted when the link
is triggered.

A typical use of this control is to display a dialog:

\qml
import QtQuick 2.4

import wesual.Ui 1.0

UiCommandLink {
    text      : "Advanced options"
    onClicked : optionsDialog.show()
}
\endqml
*/

Item {
    id : uiCommandLink

    /*!
    \qmlproperty text UiCommandLink::text
    \brief The displayed link text.

    \elideright
    */
    property alias text : linkText.text

    /*!
    \brief This signal is emitted when the link is triggered.

    The UiCommandLink can be triggered by the following user actions:

    \list
    \li Left-click
    \li Space key
    \li Enter key
    \li Return key
    \endlist
    */
    signal clicked()

    implicitWidth    : linkText.implicitWidth
    implicitHeight   : linkText.implicitHeight
                       + p_.lineWidth + p_.spacing + p_.marginTop
    activeFocusOnTab : true
    baselineOffset   : linkText.y + linkText.baselineOffset

    states : [
        State {
            name : "disabled"
            when : !uiCommandLink.enabled

            PropertyChanges {
                target : linkText
                color  : UiColors.getColor(UiColors.MediumLightGrey27)
            }
        },
        State {
            name : "highlight"
            when : p_.highlight || mouseArea.pressed

            PropertyChanges {
                target : linkText
                color  : UiColors.getColor(UiColors.LinkPressBlue)
            }
            PropertyChanges {
                target  : line
                opacity : 1
                color   : UiColors.getColor(UiColors.LinkPressBlue)
            }
        },
        State {
            name : "hovered"
            when : mouseArea.containsMouse || uiCommandLink.activeFocus

            PropertyChanges {
                target  : line
                opacity : 1
            }
        }
    ]

    QtObject {
        id : p_

        property bool highlight : false

        readonly property int horizontalMargin : 3
        readonly property int verticalMargin   : 2
        readonly property int lineWidth        : 2
        readonly property int spacing          : 4
        readonly property int marginTop        : 2
    }


    UiFocusRect {
        width  : linkText.width  + 2 * p_.horizontalMargin
        height : linkText.height + 2 * p_.verticalMargin
        y      : p_.marginTop - 1
        x      : -p_.horizontalMargin
        target : uiCommandLink
    }

    Text {
        id : linkText

        y        : p_.marginTop
        font     : UiFonts.getFont(UiFonts.Regular, 14)
        wrapMode : Text.NoWrap
        elide    : Text.ElideRight
        width    : uiCommandLink.width
        color    : UiColors.getColor(UiColors.ButtonPressBlue)
    }

    Rectangle {
        id : line

        anchors {
            top       : linkText.baseline
            topMargin : p_.spacing
            left      : uiCommandLink.left
            right     : uiCommandLink.right
        }
        height  : p_.lineWidth
        color   : UiColors.getColor(UiColors.ButtonPressBlue)
        opacity : 0
    }

    MouseArea {
        id : mouseArea

        anchors.fill : parent
        cursorShape  : Qt.PointingHandCursor
        hoverEnabled : true

        onClicked : {
            uiCommandLink.clicked();
        }
    }

    Keys.onPressed : {
        switch (event.key) {
        case Qt.Key_Space:
        case Qt.Key_Return:
        case Qt.Key_Enter:
            p_.highlight = true;
            break;
        default:
            break;
        }
    }

    Keys.onReleased : {
        switch (event.key) {
        case Qt.Key_Space:
        case Qt.Key_Return:
        case Qt.Key_Enter:
            p_.highlight = false;
            uiCommandLink.clicked();
            break;
        default:
            break;
        }
    }
}
